/** @file   door.h
 * @brief   Declaration of Door - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:41 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_DOOR_H
#define H_WWW_DOOR_H

#include <vector>
#include <iostream>
#include "www_map.h"
#include "eng2d.h"

namespace WeWantWar {

/** @class  Door
 * @brief   The doors are instances of this class.
 * @author  Tomi Lamminsaari
 * 
 * The doors in the playfield are instances of this class. They can be opened
 * by touching them and closed after a certain delay.
 *
 * The doors are usually created during the loading of a map but it is
 * possible to add new doors dynamically. The initial positions of the
 * doors are placed in the second tile layer of the map.
 * 
 * There are several different doors but so far only small doors (2 tiles wide)
 * has been implemented.
 */
class Door
{
public:
  ///
  /// Static members and methods
  /// ==========================
  
  /** Tilenumbers of the upper part of vertical door we use when animating
   * the door.
   */
  static std::vector<int> SMALL_VUPPER_DOOR_BLOCKS;
  /** Tilenumbers of the lower part of vertical door we use when animating
   * the door.
   */
  static std::vector<int> SMALL_VLOWER_DOOR_BLOCKS;
  /** Tilenumbers of the left part of vertical door we use when animating
   * the door.
   */
  static std::vector<int> SMALL_HLEFT_DOOR_BLOCKS;
  /** Tilenumbers of the right part of vertical door we use when animating
   * the door.
   */
  static std::vector<int> SMALL_HRIGHT_DOOR_BLOCKS;
  
  
  /** Reads the tilenumbers from given mapfile. This method expects that
   * the opening DOORS - tag has been read away from the stream.
   * @param     rIn               The input stream
   * @return    Nonzero if loading fails.
   */
  static int initDoorBlocks( std::istream& rIn );
  
  /** Reads the settings of Small vertical Door
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  static int initSmallVDoor( std::istream& rIn );
  
  /** Reads the settings of Small Horizontal Door
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  static int initSmallHDoor( std::istream& rIn );
  

  /** Datatype for orientation of the Door. */
  typedef int Orientation;
  /** ID of vertical door. */
  static const Orientation VERTICAL = 0;
  /** ID of horizontal door. */
  static const Orientation HORIZONTAL = 1;
  
  /** Datatype that describes the type of this Door. */
  typedef int Type;
  /** SMALL-door consist of two door blocks. */
  static const Type SMALL = 0;
  /** MEDIUM-door has not been implemented! */
  static const Type MEDIUM = 1;
  /** BIG-door has not been implemented! */
  static const Type BIG = 2;
  
  /** Key defines the type of the key player needs to open this door. */
  typedef int Key;
  /** Door don't need key. */
  static const Key NO_KEY = 0;
  /** Door need yellow-key. */
  static const Key YELLOW_KEY = 1;
  /** Door needs blue-key */
  static const Key BLUE_KEY = 2;


  /** How long each of the door-frames are visible */
  static const int DOOR_FRAME_DELAY = 4;
  
  /** How long the door stays open until it closes again */
  static const int STAYOPEN_DELAY = 120;

  /** States of the door.
   */
  enum State {
    /** Door is closed */
    STATE_CLOSED,
    /** Door is closing */
    STATE_CLOSING,
    /** Door is opening */
    STATE_OPENING,
    /** Door is entirely open */
    STATE_OPEN
  };
  

  /** Tells if block at (x,y) is a door.
   * @param     x                 X-coordinate in blocks.
   * @param     y                 Y-coordinate in blocks.
   * @return    'true' if block at (x,y) is a door-block
   */
  static bool isDoorBlock(int x, int y);
  
  /** Tells if given block is a door block.
   * @param     pB                Pointer to block.
   * @return    'true' if given block is a door-block.
   */
  static bool isDoorBlock( Map::Block* pB );
  
  /** Tells the orientation of the door at (x,y)
   * @param     x                 X-coordinate in blocks
   * @param     y                 Y-coordinate in blocks
   * @return    Orientation of the door at (x,y)
   */
  static Orientation getOrientation( int x, int y );



  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructor. The orientation depends on the block that can be found
   * at (x,y).
   * @param     t                 Type of the door.
   * @param     x                 X-coordinate of the door
   * @param     y                 Y-coordinate of the door
   */
  Door(Type t, int x, int y);
  
  
  /** Destructor. */
  ~Door();
  
private:

  /** Copy constructor. */
  Door(const Door& rO);
  
  /** Operator = */
  Door& operator = (const Door& rO);

public:


  ///
  /// Methods
  /// ==============

  /** Updates the door animation.
   */
  void update();
  
  /** Opens this door (starts the door-opening animation)
   */
  void open();

  /** Closes the door. Plays the door opening animation backwards. If there
   * is someone in the doorway we don't close it.
   */
  void close();


  ///
  /// Getter-methods
  /// ==============
  
  /** Returns the x-coordinate of the centerpoint of this door.
   * @return    X-coordinate of the centerpoint of this door (in pixels)
   */
  float getMidX() const;
  
  /** Returns the y-coordinate of the centerpoint of this door.
   * @return    Y-coordinate of the centerpoint of this door (in pixels)
   */
  float getMidY() const;
  
  /** Returns the door-rectangle that defines the area the door is
   * located.
   * @return    Rectangle that defines the door area.
   */
  eng2d::Rect2D getDoorArea() const;
  
  /** Returns the state this door is in.
   * @return    Current state
   */
  State state() const;
  

protected:


  ///
  /// Protected methods
  /// =================
  
  /** Updates the opening-animation of small door.
   */
  void updateSmallDoor();
  
  /** Animates the small door.
   */
  void animateSmallDoor();
  
  
  /** Reads the tileanimation frames as long as given ending tag has been found
   * @param     rIn               The input stream
   * @param     rVec              The vector where the tilenumbers will be
   *                              pushed.
   * @param     endTag            The ending tag
   * @return    Nonzero if fails.
   */
  static int storeFrameTable( std::istream& rIn, std::vector<int>& rVec,
                              const std::string endTag );


  ///
  /// Members
  /// =======

  /** X-coordinate (in blocks) of the upper left corner of this door. */
  int         m_posX;
  /** Y-coordinate (in blocks) of the upper left corner of this door. */
  int         m_posY;
  /** The orientation of this door. */
  Orientation m_orientation;

  /** Type of this door */
  Type        m_type;
  
  /** A counter that times the framelength while door is opening. */
  int         m_counter;
  /** Index of the current frame */
  int         m_frame;
  
  State       m_state;
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: door.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:54+02  lamminsa
 * Initial revision
 *
 */
 

